//*************************************************************************************************
//
//	Description:
//		Distortion - Shader for use with meshes that need to distort the scene
//
//	<P> Copyright (c) 2007 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Alastair Murray
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		AMurray			29/10/2007  0.1			Created
//	<TABLE>
//
//*************************************************************************************************

#include "stddefs.fxh"
#include "specialisation_globals.fxh"



//-----------------------------------------------------------------------
//
// Input parameters
//




//
// Transforms
//
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;


//
// Variables
//

float horizScroll
<
	string UIName = "Horizontal Scroll";
	float UIMin = -1.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 0.0f;

float vertScroll
<
	string UIName = "Vertical Scroll";
	float UIMin = -1.0f;
	float UIMax = 1.0f;
	bool appEdit = true;
> = 0.1f;

float distortIntensity
<
	string UIName = "Distort Intensity";
	float UIMin = 0.0f;
	float UIMax = 10.0f;
	bool appEdit = true;
> = 0.01f;

float currentTime : TIME
<
> = 0.0f;




//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	bool export = false;
> = 0;

#endif


//
// Textures
//

texture normalTexture : TEXTURE
<
	string UIName = "Normal Tex {UV1}";
	bool appEdit = true;
>;

texture distortionTexture : TEXTURE
<
	bool appEdit = false;
>;

float2 distortionTextureWidthHeight
<
	bool appEdit = false;
>;






//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D normalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
#endif
};

sampler2D distortionMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="distortionTexture"; 
	string MinFilter = "Point";
	string MagFilter = "Point";
	string MipFilter = "None";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < distortionTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
	AddressU  = Clamp;
	AddressV  = Clamp;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
#endif
};


//-----------------------------------------------------------------------
//
// Functions
//




//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;
	float3 normal : NORMAL;
	float2 texCoord : TEXCOORD0;
};


// Output structure
struct VSOUTPUT
{
	float4 position	: POSITION;														// View-coords position
	float2 texCoord	: TEXCOORD0;													// UV coords for texture channel 0
	float2 texCoordScroll : TEXCOORD1;
	float4 screenSpaceTexCoord : TEXCOORD2;											
	float3 viewSpaceNormal : TEXCOORD3;
	float2 texCoordScroll2 : TEXCOORD4;
};



//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT DistortionVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

	// Copy simple invariant params to output structure
	_output.texCoord = _input.texCoord;

	// offset the coords with time
	_output.texCoordScroll = _output.texCoord;
	_output.texCoordScroll.x += horizScroll * currentTime;
	_output.texCoordScroll.y += vertScroll * currentTime;

	_output.texCoordScroll2 = _output.texCoord + float2(0.24f,0.175f);
	_output.texCoordScroll2.x += horizScroll * currentTime*0.35f;
	_output.texCoordScroll2.y += vertScroll * currentTime*0.65f;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// calc screen space tex coords
	_output.screenSpaceTexCoord = _output.position;

	_output.viewSpaceNormal = /*normalize*/( mul( _input.normal, (float3x3)worldviewproj ) );

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

PSOUTPUT DistortionFragmentShader( VSOUTPUT _input )
{
	PSOUTPUT _output;

	// get normal
	float3 normalTexColour = tex2D( normalMap, _input.texCoordScroll );
	float3 normalTexColour2 = tex2D( normalMap, _input.texCoordScroll2 );
	float3 normal = (normalTexColour*2.0f)-1.0f;
	float3 normal2 = (normalTexColour2*2.0f)-1.0f;
	normal.xy += normal2.xy;
	normal = normalize( normal );
	float2 offset = normal.xy;
	
	// calc projected screen space coords
	float2 distortTexCoords;
	distortTexCoords.x = _input.screenSpaceTexCoord.x / _input.screenSpaceTexCoord.w;
	distortTexCoords.y = _input.screenSpaceTexCoord.y / _input.screenSpaceTexCoord.w;
	distortTexCoords = (distortTexCoords+1.0f)*0.5f;
	distortTexCoords.y = 1.0f-distortTexCoords.y;
	distortTexCoords.xy += 0.5f/distortionTextureWidthHeight;	// center pixel

	// distort the coords
	float	distanceFade = 1.0f-saturate(_input.screenSpaceTexCoord.z/200.0f);		// reduce the effect with distance
	float	distortAmount = (distortIntensity*distanceFade);
	distortTexCoords += offset*distortAmount;

	// sample distortion map
	_output.Colour.rgb = tex2D( distortionMap, distortTexCoords ).rgb;
	_output.Colour.a = 1.0f;

	// darken a bit - but don't darken on bits that don't face the camera
	_input.viewSpaceNormal = normalize( _input.viewSpaceNormal );
	float	darken_due_to_normal = (_input.viewSpaceNormal.z+1.0f)*0.5f;
	darken_due_to_normal = 1.0f - darken_due_to_normal;
	darken_due_to_normal *= darken_due_to_normal;
	darken_due_to_normal *= darken_due_to_normal;
	float	darken = 1.0f - saturate( darken_due_to_normal*((normal.x*normal.x)+(normal.y*normal.y))*0.5f );
	_output.Colour.rgb *= max( 0.7f, darken );

#ifdef _3DSMAX_
	// in Max, just show the normal map
	_output.Colour.rgb = lerp( float3(0.0f,0.0f,0.0f), (normal+1.0f)*0.5f, 1.0f );
#endif


	return _output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique Distortion
<
	bool supportsSpecialisedLighting = false;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "Distortion";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_DONT_RENDER";
	string zprimeDOFBehaviour	= "ERMB_DONT_RENDER";
	string shadowGenBehaviour = "ERMB_DONT_RENDER";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = false;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)		
		VertexShader = compile sce_vp_rsx DistortionVertexShader();
		PixelShader = compile sce_fp_rsx DistortionFragmentShader();
#else		
		VertexShader = compile vs_3_0 DistortionVertexShader();
		PixelShader = compile ps_3_0 DistortionFragmentShader();
#endif		
	}
}


